/**
 * Events store - manages event data with file-based persistence
 * In production, this would be replaced with a proper database
 */

import { Event, EventType } from '@/types/event';

// Sample events data representing real Middle East events
// This will be replaced/augmented with FindAll API data
export const sampleEvents: Event[] = [
  {
    id: 'evt-001',
    name: 'GITEX Global 2024',
    date: '2024-10-14',
    endDate: '2024-10-18',
    attendeesRange: '180,000+',
    venue: 'Dubai World Trade Centre',
    city: 'Dubai',
    country: 'UAE',
    ticketAvailable: true,
    pricing: 'From AED 350',
    description: 'The world\'s largest tech and startup event, bringing together the global tech community.',
    attendeeProfiles: 'Tech executives, CTOs, startup founders, investors, government officials, developers',
    sponsors: ['Microsoft', 'Google', 'AWS', 'Huawei', 'Dell Technologies'],
    eventType: 'exhibition',
    websiteUrl: 'https://www.gitex.com',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-10-01T00:00:00Z',
  },
  {
    id: 'evt-002',
    name: 'Saudi Entertainment Summit',
    date: '2024-11-20',
    endDate: '2024-11-22',
    attendeesRange: '5,000-8,000',
    venue: 'Riyadh Front Exhibition Center',
    city: 'Riyadh',
    country: 'Saudi Arabia',
    ticketAvailable: true,
    pricing: 'SAR 500 - SAR 2,000',
    description: 'Premier gathering for entertainment industry leaders shaping Saudi Arabia\'s entertainment sector.',
    attendeeProfiles: 'Entertainment executives, content creators, media professionals, investors',
    sponsors: ['Saudi Entertainment Ventures', 'MBC Group', 'Rotana'],
    eventType: 'summit',
    websiteUrl: 'https://saudientsummit.com',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-10-15T00:00:00Z',
  },
  {
    id: 'evt-003',
    name: 'Abu Dhabi Grand Prix',
    date: '2024-12-06',
    endDate: '2024-12-08',
    attendeesRange: '150,000+',
    venue: 'Yas Marina Circuit',
    city: 'Abu Dhabi',
    country: 'UAE',
    ticketAvailable: true,
    pricing: 'From AED 695',
    description: 'Formula 1 season finale featuring world-class racing and entertainment at Yas Marina.',
    attendeeProfiles: 'Motorsport enthusiasts, luxury travelers, corporate guests, celebrities',
    sponsors: ['Etihad Airways', 'Emirates', 'Rolex', 'Pirelli'],
    eventType: 'sports',
    websiteUrl: 'https://www.yasmarinacircuit.com',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-11-01T00:00:00Z',
  },
  {
    id: 'evt-004',
    name: 'World Future Energy Summit',
    date: '2025-01-14',
    endDate: '2025-01-16',
    attendeesRange: '30,000+',
    venue: 'ADNEC',
    city: 'Abu Dhabi',
    country: 'UAE',
    ticketAvailable: true,
    pricing: 'Free registration',
    description: 'Leading global event accelerating sustainable development and clean energy transition.',
    attendeeProfiles: 'Energy executives, sustainability leaders, government officials, investors',
    sponsors: ['Masdar', 'ADNOC', 'IRENA', 'Siemens Energy'],
    eventType: 'conference',
    websiteUrl: 'https://www.worldfutureenergysummit.com',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-11-15T00:00:00Z',
  },
  {
    id: 'evt-005',
    name: 'Qatar International Food Festival',
    date: '2025-02-10',
    endDate: '2025-02-20',
    attendeesRange: '200,000+',
    venue: 'Katara Cultural Village',
    city: 'Doha',
    country: 'Qatar',
    ticketAvailable: true,
    pricing: 'Free entry',
    description: 'Annual celebration of global cuisines featuring renowned chefs and culinary experiences.',
    attendeeProfiles: 'Food enthusiasts, families, tourists, culinary professionals',
    sponsors: ['Qatar Tourism', 'Qatar Airways', 'Marriott'],
    eventType: 'festival',
    websiteUrl: 'https://www.qiff.qa',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-12-01T00:00:00Z',
  },
  {
    id: 'evt-006',
    name: 'Arab Health Exhibition',
    date: '2025-01-27',
    endDate: '2025-01-30',
    attendeesRange: '55,000+',
    venue: 'Dubai World Trade Centre',
    city: 'Dubai',
    country: 'UAE',
    ticketAvailable: true,
    pricing: 'Free for healthcare professionals',
    description: 'The Middle East\'s largest healthcare exhibition and congress connecting medical professionals worldwide.',
    attendeeProfiles: 'Healthcare professionals, medical device manufacturers, hospital administrators, researchers',
    sponsors: ['Philips', 'GE Healthcare', 'Siemens Healthineers', 'Medtronic'],
    eventType: 'exhibition',
    websiteUrl: 'https://www.arabhealthonline.com',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-11-20T00:00:00Z',
  },
  {
    id: 'evt-007',
    name: 'Saudi Cup Horse Racing',
    date: '2025-02-22',
    endDate: '2025-02-22',
    attendeesRange: '50,000+',
    venue: 'King Abdulaziz Racetrack',
    city: 'Riyadh',
    country: 'Saudi Arabia',
    ticketAvailable: true,
    pricing: 'From SAR 100',
    description: 'The world\'s richest horse race with $20 million prize money attracting elite thoroughbreds.',
    attendeeProfiles: 'Equestrian enthusiasts, high-net-worth individuals, racing professionals, tourists',
    sponsors: ['Saudi Entertainment Authority', 'Flynas', 'SAUDIA'],
    eventType: 'sports',
    websiteUrl: 'https://www.thesaudicup.com.sa',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-12-01T00:00:00Z',
  },
  {
    id: 'evt-008',
    name: 'LEAP Tech Conference',
    date: '2025-02-09',
    endDate: '2025-02-12',
    attendeesRange: '170,000+',
    venue: 'Riyadh Exhibition & Convention Center',
    city: 'Riyadh',
    country: 'Saudi Arabia',
    ticketAvailable: true,
    pricing: 'SAR 200 - SAR 5,000',
    description: 'Global technology platform showcasing the future of tech innovation in the Kingdom.',
    attendeeProfiles: 'Tech leaders, startup founders, investors, government officials, developers',
    sponsors: ['stc', 'Neom', 'Microsoft', 'Oracle', 'Cisco'],
    eventType: 'conference',
    websiteUrl: 'https://www.onegiantleap.com',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-12-05T00:00:00Z',
  },
  {
    id: 'evt-009',
    name: 'Dubai Shopping Festival',
    date: '2024-12-15',
    endDate: '2025-01-26',
    attendeesRange: '5,000,000+',
    venue: 'Citywide',
    city: 'Dubai',
    country: 'UAE',
    ticketAvailable: true,
    pricing: 'Free',
    description: 'The Middle East\'s biggest shopping and entertainment festival with massive discounts and prizes.',
    attendeeProfiles: 'Shoppers, tourists, families, bargain hunters, luxury seekers',
    sponsors: ['Emirates', 'Dubai Tourism', 'Majid Al Futtaim', 'Emaar'],
    eventType: 'festival',
    websiteUrl: 'https://www.visitdubai.com/dsf',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-12-01T00:00:00Z',
  },
  {
    id: 'evt-010',
    name: 'Bahrain International Music Festival',
    date: '2025-03-15',
    endDate: '2025-03-22',
    attendeesRange: '30,000+',
    venue: 'Various Venues',
    city: 'Manama',
    country: 'Bahrain',
    ticketAvailable: true,
    pricing: 'BHD 15 - BHD 150',
    description: 'Annual celebration of world music featuring international and regional artists.',
    attendeeProfiles: 'Music lovers, cultural enthusiasts, expats, tourists',
    sponsors: ['Bahrain Tourism', 'Batelco', 'Gulf Air'],
    eventType: 'concert',
    websiteUrl: 'https://www.bahrainfestivals.com',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-12-10T00:00:00Z',
  },
  {
    id: 'evt-011',
    name: 'Cairo ICT',
    date: '2024-11-17',
    endDate: '2024-11-20',
    attendeesRange: '100,000+',
    venue: 'Egypt International Exhibition Center',
    city: 'Cairo',
    country: 'Egypt',
    ticketAvailable: true,
    pricing: 'Free registration',
    description: 'North Africa\'s leading technology trade fair driving digital transformation.',
    attendeeProfiles: 'IT professionals, business leaders, government officials, startups',
    sponsors: ['Orange Egypt', 'Vodafone', 'Huawei', 'Dell'],
    eventType: 'trade-show',
    websiteUrl: 'https://www.cairoict.com',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-10-20T00:00:00Z',
  },
  {
    id: 'evt-012',
    name: 'Kuwait Investment Forum',
    date: '2025-03-05',
    endDate: '2025-03-06',
    attendeesRange: '2,000+',
    venue: 'Kuwait International Fairground',
    city: 'Kuwait City',
    country: 'Kuwait',
    ticketAvailable: true,
    pricing: 'KWD 50 - KWD 500',
    description: 'Premier investment platform connecting global investors with Kuwait\'s emerging opportunities.',
    attendeeProfiles: 'Investors, fund managers, business executives, government officials',
    sponsors: ['Kuwait Investment Authority', 'National Bank of Kuwait', 'Agility'],
    eventType: 'summit',
    websiteUrl: 'https://www.kuwaitinvestmentforum.com',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-12-01T00:00:00Z',
  },
  {
    id: 'evt-013',
    name: 'Oman Tourism Summit',
    date: '2025-04-08',
    endDate: '2025-04-10',
    attendeesRange: '3,000+',
    venue: 'Oman Convention & Exhibition Centre',
    city: 'Muscat',
    country: 'Oman',
    ticketAvailable: true,
    pricing: 'OMR 25 - OMR 200',
    description: 'Strategic forum shaping the future of Oman\'s tourism and hospitality sector.',
    attendeeProfiles: 'Tourism executives, hotel operators, travel agents, government officials',
    sponsors: ['Oman Tourism', 'Oman Air', 'Marriott', 'IHG'],
    eventType: 'summit',
    websiteUrl: 'https://www.omantourismsummit.com',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-12-05T00:00:00Z',
  },
  {
    id: 'evt-014',
    name: 'Intersec Dubai',
    date: '2025-01-14',
    endDate: '2025-01-16',
    attendeesRange: '45,000+',
    venue: 'Dubai World Trade Centre',
    city: 'Dubai',
    country: 'UAE',
    ticketAvailable: true,
    pricing: 'Free for trade visitors',
    description: 'The world\'s leading trade fair for security, safety, and fire protection.',
    attendeeProfiles: 'Security professionals, fire safety experts, law enforcement, government agencies',
    sponsors: ['Honeywell', 'Hikvision', 'Bosch', 'Axis Communications'],
    eventType: 'trade-show',
    websiteUrl: 'https://www.intersec.ae.messefrankfurt.com',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-11-25T00:00:00Z',
  },
  {
    id: 'evt-015',
    name: 'Jordan Fashion Week',
    date: '2025-03-20',
    endDate: '2025-03-24',
    attendeesRange: '5,000+',
    venue: 'King Hussein Convention Center',
    city: 'Amman',
    country: 'Jordan',
    ticketAvailable: true,
    pricing: 'JOD 30 - JOD 200',
    description: 'Celebrating Arab fashion with designers from across the MENA region.',
    attendeeProfiles: 'Fashion designers, buyers, influencers, media, fashion enthusiasts',
    sponsors: ['Jordan Tourism Board', 'Arab Fashion Council'],
    eventType: 'exhibition',
    websiteUrl: 'https://www.jordanfashionweek.com',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-12-01T00:00:00Z',
  },
  {
    id: 'evt-016',
    name: 'Dubai Fintech Summit',
    date: '2025-05-06',
    endDate: '2025-05-07',
    attendeesRange: '8,000+',
    venue: 'Madinat Jumeirah',
    city: 'Dubai',
    country: 'UAE',
    ticketAvailable: true,
    pricing: 'AED 1,500 - AED 5,000',
    description: 'Premier fintech gathering connecting global financial technology leaders and innovators.',
    attendeeProfiles: 'Fintech founders, bankers, investors, regulators, technology executives',
    sponsors: ['Dubai International Financial Centre', 'Mastercard', 'Visa', 'HSBC'],
    eventType: 'conference',
    websiteUrl: 'https://www.dubaifintechsummit.com',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-12-10T00:00:00Z',
  },
  {
    id: 'evt-017',
    name: 'Riyadh Season',
    date: '2024-10-20',
    endDate: '2025-03-01',
    attendeesRange: '15,000,000+',
    venue: 'Multiple Zones Citywide',
    city: 'Riyadh',
    country: 'Saudi Arabia',
    ticketAvailable: true,
    pricing: 'Varies by event',
    description: 'The Kingdom\'s largest entertainment festival featuring concerts, shows, and attractions.',
    attendeeProfiles: 'Families, young adults, tourists, entertainment seekers',
    sponsors: ['General Entertainment Authority', 'Saudi Tourism', 'Aramco'],
    eventType: 'festival',
    websiteUrl: 'https://www.riyadhseason.com',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-10-15T00:00:00Z',
  },
  {
    id: 'evt-018',
    name: 'Middle East Film & Comic Con',
    date: '2025-03-07',
    endDate: '2025-03-09',
    attendeesRange: '50,000+',
    venue: 'Dubai World Trade Centre',
    city: 'Dubai',
    country: 'UAE',
    ticketAvailable: true,
    pricing: 'AED 75 - AED 250',
    description: 'The region\'s largest pop culture event celebrating film, comics, gaming, and cosplay.',
    attendeeProfiles: 'Pop culture fans, gamers, cosplayers, comic collectors, movie enthusiasts',
    sponsors: ['Dubai Tourism', 'Funko', 'PlayStation', 'Marvel'],
    eventType: 'exhibition',
    websiteUrl: 'https://www.mefcc.com',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-12-01T00:00:00Z',
  },
  {
    id: 'evt-019',
    name: 'Gulfood',
    date: '2025-02-17',
    endDate: '2025-02-21',
    attendeesRange: '100,000+',
    venue: 'Dubai World Trade Centre',
    city: 'Dubai',
    country: 'UAE',
    ticketAvailable: true,
    pricing: 'Free for trade visitors',
    description: 'The world\'s largest annual food and beverage trade show.',
    attendeeProfiles: 'Food industry professionals, distributors, retailers, hoteliers, chefs',
    sponsors: ['Dubai Municipality', 'Nestle', 'Unilever', 'PepsiCo'],
    eventType: 'trade-show',
    websiteUrl: 'https://www.gulfood.com',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-11-30T00:00:00Z',
  },
  {
    id: 'evt-020',
    name: 'Qatar Economic Forum',
    date: '2025-05-13',
    endDate: '2025-05-15',
    attendeesRange: '2,500+',
    venue: 'The Ritz-Carlton Doha',
    city: 'Doha',
    country: 'Qatar',
    ticketAvailable: false,
    pricing: 'Invitation only',
    description: 'High-level economic dialogue featuring global leaders and decision makers.',
    attendeeProfiles: 'CEOs, government leaders, economists, investors, policy makers',
    sponsors: ['Qatar Investment Authority', 'Bloomberg', 'Qatar Foundation'],
    eventType: 'summit',
    websiteUrl: 'https://www.qatareconomicforum.com',
    createdAt: '2024-01-01T00:00:00Z',
    updatedAt: '2024-12-05T00:00:00Z',
  },
];

// In-memory store for events
let eventsStore: Event[] = [...sampleEvents];

/**
 * Get all events from the store
 */
export function getAllEvents(): Event[] {
  return [...eventsStore];
}

/**
 * Get a single event by ID
 */
export function getEventById(id: string): Event | undefined {
  return eventsStore.find(e => e.id === id);
}

/**
 * Add new events to the store (used after FindAll API calls)
 */
export function addEvents(events: Event[]): void {
  // Deduplicate by ID
  const existingIds = new Set(eventsStore.map(e => e.id));
  const newEvents = events.filter(e => !existingIds.has(e.id));
  eventsStore = [...eventsStore, ...newEvents];
}

/**
 * Replace all events in the store
 */
export function setEvents(events: Event[]): void {
  eventsStore = [...events];
}

/**
 * Clear all events from the store
 */
export function clearEvents(): void {
  eventsStore = [];
}

/**
 * Get unique countries from events
 */
export function getCountries(): string[] {
  const countries = new Set(eventsStore.map(e => e.country));
  return Array.from(countries).sort();
}

/**
 * Get unique event types from events
 */
export function getEventTypes(): EventType[] {
  const types = new Set(eventsStore.map(e => e.eventType));
  return Array.from(types) as EventType[];
}

